/**
 *
 * \file        nvl.h
 *
 * \brief       Nonvolatile memory class
 *              This handles interfacing to the EEPROM and periodically writing data
 *              according to the timer and flag
 *
 * \author      Adolfo Velasco
 *
 * \date        07/17/08
 *
 */

#include <stdio.h>
#include "nutypedefs.h"
#include "nvl.h"
#include "os.h"

//Global variables
nvl* g_pNvlMemory = NULL;

//Local functions
void NvlTimerCallback( UINT32 lId );

/**
 * \author    Adolfo Velasco
 *
 * \brief     Initializes the NVL memory class and timer
 *
 * \date      07/17/08
 *
 * \param     pConfig
 *
 * \return    void
 *
 * \retval    none
**/
void InitNvl( const nvl_config* pConfig )
{
    //Verify parameters
    if ( !pConfig )
        return;

    //Create a new NVL class object
    g_pNvlMemory = new nvl(pConfig);
}

/**
 * \author    Adolfo Velasco
 *
 * \brief     Callback function for NVL timer task
 *
 * \date      07/17/08
 *
 * \param     lId
 *
 * \return    void
 *
 * \retval    none
**/
void NvlTimerCallback( UINT32 lId )
{
    //Check for a parameter save
    g_pNvlMemory->CheckParameterSave();
}

/**
 * \author    Adolfo Velasco
 *
 * \brief     Triggers a parameter save
 *
 * \date      07/17/08
 *
 * \param     pConfig
 *
 * \return    void
 *
 * \retval    none
**/
void DmTriggerParameterSave( void )
{
    //Set the parameter save flag
	if (g_pNvlMemory)
		g_pNvlMemory->SetParameterSave();
}

/**
 * \author        Adolfo Velasco
 *
 * \brief         Constructor for NVL class
 *
 * \date          07/17/08
 *
 * \param         pConfig - Pointer to NVL configuration table
 *
 * \return        void
 *
 * \retval        none
**/
nvl::nvl( const nvl_config* pConfig )
{
    //Save the pointer to the configuration table
    m_pConfig = pConfig;

    //Create the EEPROM class
    m_pEEPROM = new i2c_eeprom(m_pConfig->pPromConfig);

    //Initialize flags
    m_bTriggerSave = false;

    //Create the parameter save timer
    OsStartTimer(NVL_SAVE_TIMER_PERIOD_MS, (void*)NvlTimerCallback, OS_TIMER_PERIODIC);
}

/**
 * \author        Adolfo Velasco
 *
 * \brief         Checks for a parameter save and performs a write if needed
 *
 * \date          07/17/08
 *
 * \param         none
 *
 * \return        void
 *
 * \retval        none
**/
void nvl::CheckParameterSave( void )
{
    if ( m_bTriggerSave )
    {
        //Clear the flag and write
        m_bTriggerSave = false;
        Save();
    }
}

/**
 * \author        Adolfo Velasco
 *
 * \brief         Triggers a write to the EEPROM
 *
 * \date          07/17/08
 *
 * \param         none
 *
 * \return        void
 *
 * \retval        none
**/
void nvl::SetParameterSave( void )
{
    //Set the flag
    m_bTriggerSave = true;
}

/**
 * \author        Adolfo Velasco
 *
 * \brief         Saves the parameter structure into the EEPROM
 *
 * \date          07/17/08
 *
 * \param         none
 *
 * \return        void
 *
 * \retval        none
**/
void nvl::Save( void )
{
    //Write to the EEPROM
    m_pEEPROM->write_block( 0, m_pConfig->pRamData, m_pConfig->lDataSize );
}

/**
 * \author        Adolfo Velasco
 *
 * \brief         Reads the EEPROM into the parameter structure
 *
 * \date          07/17/08
 *
 * \param         none
 *
 * \return        void
 *
 * \retval        none
**/
void nvl::Load( void )
{
    //Read the EEPROM
    m_pEEPROM->read_block( 0, m_pConfig->pRamData, m_pConfig->lDataSize );
}


